<?php
/*--------------------------------------------------------------
   CreateProductVariantsAction.php 2022-02-04
   Gambio GmbH
   http://www.gambio.de
   Copyright (c) 2022 Gambio GmbH
   Released under the GNU General Public License (Version 2)
   [http://www.gnu.org/licenses/gpl-2.0.html]
 -------------------------------------------------------------*/

declare(strict_types=1);

namespace Gambio\Api\Modules\ProductVariant\App\Actions;

use Gambio\Admin\Modules\ProductVariant\Model\Exceptions\InsertionOfProductVariantsFailed;
use Gambio\Admin\Modules\ProductVariant\Model\Exceptions\ProductVariantCombinationAlreadyExists;
use Gambio\Admin\Modules\ProductVariant\Services\ProductVariantFactory;
use Gambio\Admin\Modules\ProductVariant\Services\ProductVariantsWriteService as ProductVariantsWriteServiceInterface;
use Gambio\Api\Application\Responses\CreateApiMetaDataTrait;
use Gambio\Api\Modules\ProductVariant\App\ProductVariantApiRequestParser;
use Gambio\Api\Modules\ProductVariant\App\ProductVariantApiRequestValidator;
use Gambio\Core\Application\Http\Request;
use Gambio\Core\Application\Http\Response;

/**
 * Class CreateProductVariantsAction
 * @package Gambio\Api\Modules\ProductVariant\App\Actions
 */
class CreateProductVariantsAction
{
    use CreateApiMetaDataTrait;
    
    /**
     * @var ProductVariantApiRequestValidator
     */
    private $validator;
    
    /**
     * @var ProductVariantApiRequestParser
     */
    private $parser;
    
    /**
     * @var ProductVariantFactory
     */
    private $factory;
    
    /**
     * @var ProductVariantsWriteServiceInterface
     */
    private $service;
    
    
    /**
     * CreateProductVariantsAction constructor.
     *
     * @param ProductVariantApiRequestValidator    $validator
     * @param ProductVariantApiRequestParser       $parser
     * @param ProductVariantFactory                $factory
     * @param ProductVariantsWriteServiceInterface $service
     */
    public function __construct(
        ProductVariantApiRequestValidator $validator,
        ProductVariantApiRequestParser $parser,
        ProductVariantFactory $factory,
        ProductVariantsWriteServiceInterface $service
    ) {
        $this->validator = $validator;
        $this->parser    = $parser;
        $this->factory   = $factory;
        $this->service   = $service;
    }
    
    
    /**
     * @param Request  $request
     * @param Response $response
     * @param array    $args
     *
     * @return Response
     */
    public function __invoke(Request $request, Response $response, array $args): Response
    {
        $errors = $this->validator->validateCreationBody($request->getParsedBody());
    
        if (empty($errors) === false) {
        
            return $response->withStatus(400)->withJson(['errors' => $errors]);
        }
        
        $creationArguments = $this->parser->parseProductVariantsData($request, $errors);
    
        if (empty($errors) === false || empty($creationArguments) === true) {
        
            return $response->withStatus(422)->withJson(['errors' => $errors]);
        }
        
        try {
    
            $variantIds = $this->service->createMultipleProductVariants(...$creationArguments);
        } catch (InsertionOfProductVariantsFailed | ProductVariantCombinationAlreadyExists $exception) {
    
            return $response->withStatus(422)->withJson(['errors' => [$exception->getMessage()]]);
        }
    
        $links   = [];
        $baseUrl = rtrim($this->parser->getResourceUrlFromRequest($request), '/');
        foreach ($variantIds as $id) {
        
            $links[] = $baseUrl . '/' . $id->value();
        }
    
        return $response->withStatus(200)->withJson(['data' => $variantIds->toArray(), '_meta' => $this->createApiMetaData($links),]);
    }
}